<?php
defined('BASEPATH') OR exit('No direct script access allowed');

/**
 * API untuk proses pendaftaran user (nomor HP + PIN, tanpa email).
 * Otomatis membuat saldo & poin awal = 0 pada tabel relasi.
 */
class Api_daftar_activity_cek extends CI_Controller
{
    public function __construct()
    {
        parent::__construct();
        $this->load->database();
    }

    /**
     * Endpoint utama pendaftaran user
     * @return void
     */
    public function index()
    {
        $this->_set_json_header();

        if (!$this->_is_post()) {
            return $this->_response('004', 'Method tidak diizinkan');
        }

        // 1. Ambil & validasi parameter
        $params = $this->_get_registration_params();
        if (!$params['success']) {
            return $this->_response('003', $params['message']);
        }
        $user = $params['data'];

        // 2. Cek apakah nomor HP sudah terdaftar
        if ($this->_is_phone_registered($user['nomor_hp'])) {
            return $this->_response('002', 'Nomor HP atau Whatsapp Anda sudah terdaftar');
        }

        // 3. Generate data untuk user baru
        $data_to_insert = $this->_build_user_data($user);

        // 4. Simpan ke database (pakai transaksi, sekalian saldo & poin)
        $this->db->trans_start();

        // Insert user
        $this->db->insert('tbl_mst_user', $data_to_insert);

        // Ambil kode user yang barusan dibuat untuk relasi saldo & poin
        $mst_user_kode = $data_to_insert['mst_user_kode'];

        // Generate kode saldo & poin
        $mst_saldo_kode = "SALDO-" . strtoupper(substr(sha1(uniqid()), 0, 10)) . "-" . rand(0, 999) . "-" . date('Hms');
        $mst_poin_kode  = "POIN-"  . strtoupper(substr(sha1(uniqid()), 0, 10)) . "-" . rand(0, 999) . "-" . date('Hms');

        // Insert saldo awal
        $this->db->insert('tbl_mst_saldo', [
            'mst_saldo_kode'      => $mst_saldo_kode,
            'mst_saldo_user_kode' => $mst_user_kode,
            'mst_saldo_jumlah'    => '0'
        ]);

        // Insert poin awal
        $this->db->insert('tbl_mst_poin', [
            'mst_poin_kode'      => $mst_poin_kode,
            'mst_poin_user_kode' => $mst_user_kode,
            'mst_poin_jumlah'    => '0'
        ]);

        $this->db->trans_complete();

        if ($this->db->trans_status() === FALSE) {
            $this->_log_db_error([
                'user'  => $data_to_insert,
                'saldo' => [
                    'mst_saldo_kode'      => $mst_saldo_kode,
                    'mst_saldo_user_kode' => $mst_user_kode,
                    'mst_saldo_jumlah'    => '0'
                ],
                'poin'  => [
                    'mst_poin_kode'      => $mst_poin_kode,
                    'mst_poin_user_kode' => $mst_user_kode,
                    'mst_poin_jumlah'    => '0'
                ]
            ]);
            return $this->_response('003', 'Pendaftaran gagal, silakan coba lagi.');
        }

        // 5. Sukses
        return $this->_response('001', 'Pendaftaran berhasil!', [
            'user_kode'  => $mst_user_kode,
            'saldo_kode' => $mst_saldo_kode,
            'poin_kode'  => $mst_poin_kode
        ]);
    }

    // =================== HELPER METHOD ===================

    /**
     * Set header response JSON
     */
    private function _set_json_header()
    {
        header('Content-Type: application/json; charset=utf-8');
    }

    /**
     * Cek method request POST
     * @return bool
     */
    private function _is_post()
    {
        return $this->input->method() === 'post';
    }

    /**
     * Ambil, dekripsi, dan validasi parameter pendaftaran
     * @return array
     */
    private function _get_registration_params()
    {
        $enc_nama     = $this->input->post('mst_user_nama_lengkap', true);
        $enc_nomor_hp = $this->input->post('mst_user_nomor_hp', true);
        $enc_pin      = $this->input->post('mst_user_pin', true);

        if (empty($enc_nama) || empty($enc_nomor_hp) || empty($enc_pin)) {
            return [
                'success' => false,
                'message' => 'Parameter pendaftaran tidak lengkap'
            ];
        }

        try {
            $nama_lengkap = trim(function_exists('decrypt') ? decrypt($enc_nama, get_key()) : $enc_nama);
            $nomor_hp     = trim(function_exists('decrypt') ? decrypt($enc_nomor_hp, get_key()) : $enc_nomor_hp);
            $pin_plain    = trim(function_exists('decrypt') ? decrypt($enc_pin, get_key()) : $enc_pin);
        } catch (Exception $e) {
            return [
                'success' => false,
                'message' => 'Data tidak valid'
            ];
        }

        if ($nama_lengkap === '' || $nomor_hp === '' || strlen($pin_plain) < 4) {
            return [
                'success' => false,
                'message' => 'Format data tidak sesuai'
            ];
        }

        return [
            'success' => true,
            'data'    => [
                'nama_lengkap' => $nama_lengkap,
                'nomor_hp'     => $nomor_hp,
                'pin'          => $pin_plain
            ]
        ];
    }

    /**
     * Cek apakah nomor HP sudah terdaftar
     * @param string $nomor_hp
     * @return bool
     */
    private function _is_phone_registered($nomor_hp)
    {
        return $this->db->where('mst_user_nomor_hp', $nomor_hp)
                        ->count_all_results('tbl_mst_user') > 0;
    }

    /**
     * Siapkan data array untuk insert user baru
     * @param array $user
     * @return array
     */
    private function _build_user_data($user)
    {
        $mst_user_kode           = $this->_generate_user_kode();
        $mst_user_tipe           = "Normal";
        $nama_stripped           = preg_replace("/\s+/", "", strtoupper($user['nama_lengkap']));
        $mst_user_referal_kode   = $this->_generate_referal_kode($nama_stripped, $user['nomor_hp']);
        $mst_user_token          = $this->_generate_token();
        $mst_user_tanggal_daftar = date('Y-m-d H:i:s');
        $mst_user_status         = "0"; // Belum aktif

        return [
            'mst_user_kode'           => $mst_user_kode,
            'mst_user_nama_lengkap'   => $user['nama_lengkap'],
            'mst_user_nomor_hp'       => $user['nomor_hp'],
            'mst_user_pin'            => password_hash($user['pin'], PASSWORD_BCRYPT),
            'mst_user_password'       => '',
            'mst_user_tipe'           => $mst_user_tipe,
            'mst_user_referal_kode'   => $mst_user_referal_kode,
            'mst_user_token'          => $mst_user_token,
            'mst_user_firebase'       => '',
            'mst_user_zona_waktu'     => '',
            'mst_user_tanggal_daftar' => $mst_user_tanggal_daftar,
            'mst_user_status'         => $mst_user_status
        ];
    }

    /**
     * Generate kode unik user
     * @return string
     */
    private function _generate_user_kode()
    {
        return "USER-" . strtoupper(substr(sha1(uniqid()), 0, 10)) . "-" . rand(0, 999) . "-" . date('Hms');
    }

    /**
     * Generate kode referral unik
     * @param string $nama_stripped
     * @param string $nomor_hp
     * @return string
     */
    private function _generate_referal_kode($nama_stripped, $nomor_hp)
    {
        return explode(" ", str_replace(" ", rand(1, 9), substr(str_shuffle(trim($nama_stripped)), 0, 4)))[0] . substr($nomor_hp, -2);
    }

    /**
     * Generate token user
     * @return string
     */
    private function _generate_token()
    {
        return "TOKEN-" . strtoupper(substr(sha1(uniqid()), 0, 10)) . "-" . rand(0, 999) . "-" . date('d-m-Y--H:i:s');
    }

    /**
     * Response JSON standar
     * @param string $status
     * @param string $message
     * @param array $extra
     * @return void
     */
    private function _response($status, $message, $extra = [])
    {
        $response = array_merge([
            'status' => $status,
            'pesan'  => $message
        ], $extra);

        echo json_encode($response);
        return;
    }

    /**
     * Logging error database
     * @param array $data
     * @return void
     */
    private function _log_db_error($data)
    {
        $db_error = $this->db->error();
        log_message('error', 'Gagal insert user/saldo/poin: ' . json_encode($db_error));
        log_message('error', 'Data insert: ' . json_encode($data));
    }
}
