<?php
// File: api/application/controllers/splash_activity/v1/Api_splash_activity.php
// Controller untuk mengembalikan konfigurasi splash screen terbaru dalam format JSON
// Dilengkapi logging sangat detail dalam Bahasa Indonesia untuk mempermudah debugging

defined('BASEPATH') OR exit('No direct script access allowed');

require APPPATH . 'libraries/RestController.php';
require APPPATH . 'libraries/Format.php';

use chriskacerguis\RestServer\RestController;

class Api_splash_activity extends RestController {

    private $startTime;

    public function __construct() {
        parent::__construct();
        $this->startTime = microtime(true);
        log_message('debug', '[SplashAPI] __construct() dimulai pada ' . date('Y-m-d H:i:s') . ', memori awal: ' . memory_get_usage());

        $this->load->model('Splash_model');
        log_message('debug', '[SplashAPI] Model "Splash_model" dimuat');

        // Header CORS
        header('Access-Control-Allow-Origin: *');
        header('Access-Control-Allow-Methods: GET, POST, OPTIONS');
        header('Access-Control-Allow-Headers: Content-Type, If-None-Match');
        log_message('debug', '[SplashAPI] Header CORS diatur');

        // Shutdown handler untuk error fatal
        register_shutdown_function(function() {
            $error = error_get_last();
            if ($error && in_array($error['type'], [E_ERROR, E_PARSE, E_CORE_ERROR, E_COMPILE_ERROR])) {
                log_message('error', '[SplashAPI] Error fatal shutdown: ' . json_encode($error));
            }
        });
    }

    /**
     * GET endpoint: splash
     */
    public function splash_get() {
        log_message('debug', '[SplashAPI] splash_get() dipanggil, method: ' . \$this->input->method());

        // Log raw input & server info
        log_message('debug', '[SplashAPI] RAW INPUT: ' . \$this->input->raw_input_stream);
        log_message('debug', '[SplashAPI] SERVER: ' . json_encode([
            'REQUEST_URI'  => \$_SERVER['REQUEST_URI'],
            'QUERY_STRING' => \$_SERVER['QUERY_STRING'],
            'REMOTE_ADDR'  => \$_SERVER['REMOTE_ADDR'] ?? ''
        ]));

        // Bersihkan buffer
        while (ob_get_level() > 0) {
            ob_end_clean();
            log_message('debug', '[SplashAPI] ob_end_clean() dipanggil');
        }

        try {
            // Ambil data terbaru dari DB
            \$config = \$this->Splash_model->get_latest();
            log_message('debug', '[SplashAPI] Data DB diperoleh: ' . json_encode(\$config));
            if (!\$config) {
                throw new Exception('Tidak ada data splash di database');
            }

            // Validasi setiap field
            foreach (['splash_version','splash_url','splash_time'] as \$f) {
                log_message('debug', "[SplashAPI] Mengecek field {\$f}: " . json_encode(\$config->\$f));
                if (!isset(\$config->\$f) || trim((string)\$config->\$f) === '') {
                    throw new Exception("Field {\$f} kosong atau tidak valid: " . json_encode(\$config));
                }
                log_message('debug', "[SplashAPI] Field {\$f} valid: {\$config->\$f}");
            }

            // Cek keberadaan file fisik
            \$filePath = FCPATH . 'admin/assets/upload/data-splash/' . \$config->splash_url;
            log_message('debug', '[SplashAPI] FCPATH: ' . FCPATH);
            log_message('debug', '[SplashAPI] Memeriksa keberadaan file: ' . \$filePath);
            if (!file_exists(\$filePath)) {
                // Log isi direktori untuk debugging
                \$dir = dirname(\$filePath);
                \$listing = @scandir(\$dir);
                log_message('error', '[SplashAPI] Daftar file di ' . \$dir . ': ' . json_encode(\$listing));
                throw new Exception('File splash tidak ditemukan: ' . \$filePath);
            }

            // Bangun URL final
            \$base = rtrim(base_url(), '/');
            log_message('debug', '[SplashAPI] base_url(): ' . \$base);
            if (substr(\$base, -4) === '/api') {
                \$base = substr(\$base, 0, -4);
                log_message('debug', '[SplashAPI] Hapus "/api", base sekarang: ' . \$base);
            }
            \$url = \$base . '/admin/assets/upload/data-splash/' . \$config->splash_url;
            log_message('debug', '[SplashAPI] URL splash: ' . \$url);

            // Siapkan response
            \$response = [
                'version'       => (int)\$config->splash_version,
                'url'           => \$url,
                'displayTimeMs' => (int)\$config->splash_time
            ];
            log_message('debug', '[SplashAPI] Response array: ' . json_encode(\$response));

            // ETAG untuk caching
            \$etag = '"' . md5(json_encode(\$response)) . '"';
            \$clientEtag = \$this->input->get_request_header('If-None-Match');
            log_message('debug', '[SplashAPI] ETag server: ' . \$etag . ', client: ' . \$clientEtag);
            if (\$clientEtag === \$etag) {
                log_message('debug', '[SplashAPI] ETag match, kirim 304');
                \$this->output->set_header('ETag: ' . \$etag);
                return \$this->response(NULL, RestController::HTTP_NOT_MODIFIED);
            }
            \$this->output->set_header('ETag: ' . \$etag);

            // Kirim data sukses
            \$duration = microtime(true) - \$this->startTime;
            log_message('debug', '[SplashAPI] Waktu eksekusi: ' . round(\$duration, 4) . ' detik');
            return \$this->response(\$response, RestController::HTTP_OK);

        } catch (Throwable \$e) {
            // Bersihkan buffer sebelum error
            while (ob_get_level() > 0) {
                ob_end_clean();
                log_message('debug', '[SplashAPI] ob_end_clean() di exception');
            }
            log_message('error', '[SplashAPI] Exception: ' . \$e->getMessage());
            log_message('error', '[SplashAPI] Stack trace: ' . \$e->getTraceAsString());

            // Kirim error ke client
            \$errorPayload = [
                'error'   => 'Server error',
                'message' => \$e->getMessage()
            ];
            log_message('debug', '[SplashAPI] Kirim 500 Internal Error');
            return \$this->response(\$errorPayload, RestController::HTTP_INTERNAL_ERROR);
        }
    }
}
